function out = Source_Code_2(params_override)


%% ---------------- Parameters ----------------
P = struct( ...
    'iAp',-120, ...             % pA
    'VL',-70,'VK',-90,'VNa',50,'VCa',80,'VH',-30, ...             % mV
    'gNa',450,'gKNa',40,'gH',4,'gA',5,'gCaT',2.65,'C',100,'kr',0.3, ...
    'gL',2,'gNap',1,'gCa',19, ...                                  % nS
    'taunbar',10,'tauhpbar',1000,'taue',20,'tauh',1, ...
    'taurs',1500,'taur0',200,'taur1',87.5, ...
    'Ca_ex',2.5,'RTF',26.7, ...
    'thetam',-35,'thetan',-30,'thetas',-20,'thetamp',-40,'thetahp',-48, ...
    'thetaa',-20,'thetae',-60,'thetarf',-105,'thetars',-105, ...
    'thetaaT',-65,'thetab',0.4,'thetarT',-67,'thetarrT',68, ...
    'sigmam',-5,'sigman',-5,'sigmas',-0.05,'sigmamp',-6,'sigmahp',6, ...
    'sigmaa',-10,'sigmae',5,'sigmarf',5,'sigmars',25, ...
    'sigmaaT',-7.8,'sigmab',-0.1,'sigmarT',2,'sigmarrT',2.2, ...
    'f',0.1,'eps',0.0015,'kCa',0.3,'bCa',0.1,'ks',0.5, ...
    'Rpump',0.0008,'Kp',15,'prf',100,'alphaNa',0.0001,'Naeq',8, ...
    'gK',1700,'gSK',1, ...                % your overrides in this variant
    'from',170,'to',870);                 % stimulus window (ms)

% Allow user overrides
if nargin==1 && ~isempty(params_override)
    fns = fieldnames(params_override);
    for k = 1:numel(fns), P.(fns{k}) = params_override.(fns{k}); end
end

%% ---------------- Initial conditions ----------------
% State order must match RHS and labels:
x0 = [ ...
   -67.5;  ... V (mV)
    0.0002;... n
    0.01;  ... h
    0.05;  ... e
    0.0;   ... rT  (not in init -> set 0)
    0.103; ... Cai
    7.09;  ... Nai
    0.03;  ... rf
    0.03;  ... rs
    0.1];  ... hp

labels = {'V','n','h','e','rT','Cai','Nai','rf','rs','hp'};

%% ---------------- Time setup ----------------
dt = 0.01;             % ms
t_eval = 0:dt:1500;    % from @ total=1500
tspan  = [t_eval(1) t_eval(end)];

%% ---------------- Solve ----------------
opts = odeset('RelTol',1e-6,'AbsTol',1e-9);
rhs = @(t,x) odeRHS(t,x,P);
[t, X] = ode45(rhs, t_eval, x0, opts); % sample exactly at dt grid

%% ---------------- Diagnostics (currents, stimulus) ------------
curr = compute_currents(t, X, P);

%% ---------------- Quicklook (optional) ------------------------
figure; plot(t, X(:,1)); xlabel('Time (ms)'); ylabel('V (mV)'); title('Membrane potential');

%% ---------------- Package output ------------------------------
out = struct('t', t, 'X', X, 'labels', {labels}, 'currents', curr, 'params', P);

end % run_STN_model_variant


%% =================== Right-hand side ==========================
function dx = odeRHS(t, x, P)
% Unpack states
V   = x(1);
n   = x(2);
h   = x(3);
e   = x(4);
rT  = x(5);
Cai = x(6);
Nai = x(7);
rf  = x(8);
rs  = x(9);
hp  = x(10);

% Helper
logi = @(z,th,sg) 1./(1 + exp((z - th)./sg));

% --- Ca-L ---
sinf  = logi(V, P.thetas, P.sigmas);
iCaL  = P.gCa .* (sinf.^2) .* V .* (P.Ca_ex ./ (1 - exp((2*V)/P.RTF)));

% --- Ca-T ---
aTinf = logi(V, P.thetaaT, P.sigmaaT);
bTinf = 1./(1 + exp((rT - P.thetab)/P.sigmab)) - 1./(1 + exp(-P.thetab/P.sigmab));
rTinf = 1./(1 + exp((V - P.thetarT)/P.sigmarT));
taurT = P.taur0 + P.taur1./(1 + exp((V - P.thetarrT)/P.sigmarrT));
iCaT  = P.gCaT .* (aTinf.^3) .* (bTinf.^3) .* V .* (P.Ca_ex ./ (1 - exp((2*V)/P.RTF)));

% --- Nap ---
mpinf = logi(V, P.thetamp, P.sigmamp);
hpinf = logi(V, P.thetahp, P.sigmahp);
tauhp = P.tauhpbar ./ cosh((V - P.thetahp)/(2*P.sigmahp));   % v->V
iNap  = P.gNap .* mpinf .* hp .* (V - P.VNa);

% --- Leak ---
iL    = P.gL .* (V - P.VL);

% --- K (delayed rectifier) ---
ninf  = logi(V, P.thetan, P.sigman);
taun  = P.taunbar ./ cosh((V - P.thetan)/(2*P.sigman));
iK    = P.gK .* (n.^4) .* (V - P.VK);

% --- Na (transient) ---
minf   = logi(V, P.thetam, P.sigmam);
alphah = 0.128 .* exp(-(V + 50)/18);
betah  = 4 ./ (1 + exp(-(V + 27)/5));
hinf   = alphah ./ (alphah + betah);
iNa    = P.gNa .* (minf.^3) .* h .* (V - P.VNa);

% --- A-type K ---
ainf = logi(V, P.thetaa, P.sigmaa);
einf = logi(V, P.thetae, P.sigmae);
iA   = P.gA .* ainf .* e .* (V - P.VK);

% --- SK (Ca-dependent K) ---
kinf = (Cai.^2) ./ (Cai.^2 + (P.ks.^2));
iSK  = P.gSK .* kinf .* (V - P.VK);

% --- KNa (Na-dependent K) ---
winf  = 0.37 ./ (1 + (38.7./Nai).^3.5);
phiNa = @(xx) (xx.^3) ./ (xx.^3 + P.Kp.^3);
iKNa  = P.gKNa .* winf .* (V - P.VK);

% --- H current ---
rfinf = 1./(1 + exp((V - P.thetarf)/P.sigmarf));
taurf = P.prf ./ ( -7.4*(V+70)./(exp(-(V+70)/0.8) - 1) + 65*exp(-(V+56)/23) ); % v->V
rsinf = 1./(1 + exp((V - P.thetars)/P.sigmars));
iH    = P.gH .* (P.kr.*rf + (1 - P.kr).*rs) .* (V - P.VH);

% --- Stimulus ---
if t < P.from
    iApp = 0;
elseif t > P.from && t < P.to
    iApp = P.iAp;
else
    iApp = 0;
end

% --- Differential equations ---
dV   = (-iL - iK - iNa - iCaL - iCaT - iA - iSK - iKNa - iH - iNap + iApp) / P.C;
dn   = (ninf - n) / taun;
dh   = (hinf - h) / P.tauh;
de   = (einf - e) / P.taue;
drT  = (rTinf - rT) / taurT;
dCai = -P.f * ( P.eps*(iCaL + iCaT) + P.kCa*(Cai - P.bCa) );
dNai = -P.alphaNa*(iNa + iNap) - 3*P.Rpump*(phiNa(Nai) - phiNa(P.Naeq));
drf  = (rfinf - rf) / taurf;
drs  = 3*(rsinf - rs) / P.taurs;    % NOTE: factor 3 per your XPP
dhp  = (hpinf - hp) / tauhp;

dx = [dV; dn; dh; de; drT; dCai; dNai; drf; drs; dhp];
end


%% =================== Currents & stimulus for diagnostics ======
function curr = compute_currents(t, X, P)
V   = X(:,1);  n = X(:,2);  h = X(:,3);  e = X(:,4);  rT = X(:,5);
Cai = X(:,6);  Nai = X(:,7); rf = X(:,8); rs = X(:,9); hp = X(:,10); %#ok<NASGU>

logi = @(z,th,sg) 1./(1 + exp((z - th)./sg));

sinf  = logi(V, P.thetas, P.sigmas);
iCaL  = P.gCa .* (sinf.^2) .* V .* (P.Ca_ex ./ (1 - exp((2*V)/P.RTF)));

aTinf = logi(V, P.thetaaT, P.sigmaaT);
bTinf = 1./(1 + exp((rT - P.thetab)/P.sigmab)) - 1./(1 + exp(-P.thetab/P.sigmab));
iCaT  = P.gCaT .* (aTinf.^3) .* (bTinf.^3) .* V .* (P.Ca_ex ./ (1 - exp((2*V)/P.RTF)));

mpinf = logi(V, P.thetamp, P.sigmamp);
iNap  = P.gNap .* mpinf .* X(:,10) .* (V - P.VNa);  % uses hp (= X(:,10))

iL    = P.gL .* (V - P.VL);

minf  = logi(V, P.thetam, P.sigmam);
iNa   = P.gNa .* (minf.^3) .* h .* (V - P.VNa);

iK    = P.gK .* (n.^4) .* (V - P.VK);

ainf  = logi(V, P.thetaa, P.sigmaa);
einf  = logi(V, P.thetae, P.sigmae);
iA    = P.gA .* ainf .* e .* (V - P.VK);

kinf  = (Cai.^2) ./ (Cai.^2 + (P.ks.^2));
iSK   = P.gSK .* kinf .* (V - P.VK);

winf  = 0.37 ./ (1 + (38.7./Nai).^3.5);
iKNa  = P.gKNa .* winf .* (V - P.VK);

iH    = P.gH .* (P.kr.*rf + (1 - P.kr).*rs) .* (V - P.VH);

iApp  = zeros(size(t));
iApp(t > P.from & t < P.to) = P.iAp;

curr = struct('iL',iL,'iK',iK,'iNa',iNa,'iCaL',iCaL,'iCaT',iCaT, ...
              'iA',iA,'iSK',iSK,'iKNa',iKNa,'iH',iH,'iNap',iNap,'iApp',iApp);
end
